// number of static variables of outer class
// (does not count inner class instance variables)

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;
import java.util.ArrayList;

public class StaticVariableCountCheck extends AbstractCheck {
    private int count;                    // number of static variables detected
    private int min = 0;                  // minimum number required
    private int max = Integer.MAX_VALUE;  // maximum number allowed

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_STATIC = "static.variable.count";
    public static final String MSG_NONFINAL_STATIC = "static.nonfinal.variable.count";

    // the static variables
    private ArrayList<String> staticVariables;

    // If true, does not apply the check to final variables.
    private boolean ignoreFinal = false;  

    @Override
    public int[] getDefaultTokens() {
        return new int[] {TokenTypes.VARIABLE_DEF};
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] {TokenTypes.VARIABLE_DEF};
    }
    @Override
    public int[] getAcceptableTokens() {
        return new int[] {TokenTypes.VARIABLE_DEF};
    }


    /**
      * Sets a maximum count.
      * @param max the maximum count.
      */
    public void setMax(int max) {
        this.max = max;
    }

    /**
      * Sets a minimum count.
      * @param min the minimum count.
      */
    public void setMin(int min) {
        this.min = min;
    }

   /**
     * Sets whether we should apply the check to final variables.
     *
     * @param ignore new value of the property.
     */
    public void setIgnoreFinal(boolean ignore) {
        ignoreFinal = ignore;
    }

    @Override
    public void visitToken(DetailAST ast) {
        if (Utilities.isStaticVariable(ast) && (!Utilities.isStaticFinalVariable(ast) || !ignoreFinal)) {
            staticVariables.add(findVariableName(ast));
            count++;
        }
    }

    @Override
    public void beginTree(DetailAST rootAST) {
        count = 0;
        staticVariables = new ArrayList<String>();
    }

    @Override
    public void finishTree(DetailAST rootAST) {
        if (count > this.max || count < this.min) {
            String finalMessage = "";
            if (ignoreFinal) finalMessage = " non-final";

            // put list of variables in parentheses instead of square brackets
            String variableList = staticVariables.toString();
            variableList = variableList.substring(1, variableList.length() - 1);
            if (ignoreFinal) {
                log(rootAST.getLineNo(),
                    MSG_NONFINAL_STATIC,
                    count,
                    variableList);
            }
            else {
                log(rootAST.getLineNo(),
                    MSG_STATIC,
                    count,
                    variableList);
            }
        }
    }

    // extract and return the variable name from the AST
    private String findVariableName(DetailAST ast) {
        DetailAST identifier = ast.findFirstToken(TokenTypes.IDENT);
        // return identifier.toString();
        return "'" + identifier.getText() + "'";
    }

}
