package edu.princeton.cs.lift.checkstyle;

import java.util.regex.Pattern;

import com.google.common.base.Optional;

import com.puppycrawl.tools.checkstyle.api.*;
import com.puppycrawl.tools.checkstyle.utils.ScopeUtil;

public class Utilities {

   // public static boolean isEqualsMethod(DetailAST ast)
   // see checkstyle/utils/CheckUtil.java

   // public static boolean isNonVoidMethod(DetailAST methodDefAst) 
   // see checkstyle/utils/CheckUtil.java


    // is the class a nested class?
    public static boolean isNestedClass(DetailAST ast) {
        if (ast.getType() != TokenTypes.CLASS_DEF) return false;
        return !ScopeUtil.isOuterMostType(ast);
    }


    // return last token of specified type; null if no such token
    public static DetailAST findLastToken(DetailAST ast, int type) {
        DetailAST returnValue = null;
        for (DetailAST x = ast.getFirstChild(); x != null; x = x.getNextSibling()) {
            if (x.getType() == type) {
                returnValue = x;
            }
        }
        return returnValue;
    }

    /**
     * Whether the type is java.lang.String.
     * @param typeAst the type to check.
     * @return true, if the type is java.lang.String.
     */
    public static boolean isStringType(DetailAST typeAst) {
        final FullIdent type = FullIdent.createFullIdent(typeAst);
        return "String".equals(type.getText())
            || "java.lang.String".equals(type.getText());
    }


    // is the variable a static variable?
    public static boolean isStaticVariable(DetailAST ast) {
        if (ast.getType() != TokenTypes.VARIABLE_DEF) return false;

        // [wayne f17] branchContains() getting deprecated
        // boolean isStatic = modifiersAST.branchContains(TokenTypes.LITERAL_STATIC);
        // boolean isFinal  = modifiersAST.branchContains(TokenTypes.FINAL);

        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        boolean isStatic = modifiersAST.findFirstToken(TokenTypes.LITERAL_STATIC) != null;
        boolean isFinal  = modifiersAST.findFirstToken(TokenTypes.FINAL) != null;

        boolean isField  = !ScopeUtil.isLocalVariableDef(ast);
        boolean isOuter  = ScopeUtil.isOuterMostType(ast.getParent().getParent());
        boolean isClass  = (ast.getParent().getParent().getType() == TokenTypes.CLASS_DEF);
        return isStatic && isField && isClass;
    }

    // is the variable a static final variable (a constant)?
    public static boolean isStaticFinalVariable(DetailAST ast) {
        if (ast.getType() != TokenTypes.VARIABLE_DEF) return false;
        boolean isStaticVariable = isStaticVariable(ast);
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // boolean isFinal = modifiersAST.branchContains(TokenTypes.FINAL);
        boolean isFinal = modifiersAST.findFirstToken(TokenTypes.FINAL) != null;
        return isStaticVariable && isFinal;
    }

    // is the variable a final instance variable (e.g., in an immutable class)?
    public static boolean isFinalInstanceVariable(DetailAST ast) {
        if (ast.getType() != TokenTypes.VARIABLE_DEF) return false;
        boolean isInstanceVariable = isInstanceVariable(ast);
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // boolean isFinal = modifiersAST.branchContains(TokenTypes.FINAL);
        boolean isFinal = modifiersAST.findFirstToken(TokenTypes.FINAL) != null;
        return isInstanceVariable && isFinal;
    }

    // is the variable an instance variable?
    public static boolean isInstanceVariable(DetailAST ast) {
        if (ast.getType() != TokenTypes.VARIABLE_DEF) return false;
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // boolean isStatic = modifiersAST.branchContains(TokenTypes.LITERAL_STATIC);
        boolean isStatic = modifiersAST.findFirstToken(TokenTypes.LITERAL_STATIC) != null;
        boolean isField  = !ScopeUtil.isLocalVariableDef(ast);
        boolean isClass  = (ast.getParent().getParent().getType() == TokenTypes.CLASS_DEF);
        return !isStatic && isField && isClass;
    }

    // is the method a static method?
    public static boolean isStaticMethod(DetailAST ast) {
        if (ast.getType() != TokenTypes.METHOD_DEF) return false;
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // return modifiersAST.branchContains(TokenTypes.LITERAL_STATIC);
        boolean isStatic = modifiersAST.findFirstToken(TokenTypes.LITERAL_STATIC) != null;
        return isStatic;
    }

    // is the method an instance method?
    public static boolean isInstanceMethod(DetailAST ast) {
        if (ast.getType() != TokenTypes.METHOD_DEF) return false;
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // return !modifiersAST.branchContains(TokenTypes.LITERAL_STATIC);
        boolean isStatic = modifiersAST.findFirstToken(TokenTypes.LITERAL_STATIC) != null;
        return !isStatic;
    }

    // is the constructor public?
    public static boolean isPublicConstructor(DetailAST ast) {
        if (!isConstructor(ast)) return false;
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // return modifiersAST.branchContains(TokenTypes.LITERAL_PUBLIC);
        boolean isPublic = modifiersAST.findFirstToken(TokenTypes.LITERAL_PUBLIC) != null;
        return isPublic;
    }

    // is the method public?
    public static boolean isPublicMethod(DetailAST ast) {
        if (ast.getType() != TokenTypes.METHOD_DEF) return false;
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // return modifiersAST.branchContains(TokenTypes.LITERAL_PUBLIC);
        boolean isPublic = modifiersAST.findFirstToken(TokenTypes.LITERAL_PUBLIC) != null;
        return isPublic;
    }

    // is the method public?
    public static boolean isConstructor(DetailAST ast) {
        return ast.getType() == TokenTypes.CTOR_DEF;
    }

    // is the ast a descendant of a node of specified type?
    public static boolean isDescendantOf(DetailAST ast, int type) {
        for (DetailAST x = ast; x != null; x = x.getParent())
            if (x.getType() == type)
                return true;
        return false;
    }

    /**
     * Returns whether a node is contained in a class definition.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isInClass(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.CLASS_DEF) return true;
            if (x.getType() == TokenTypes.INTERFACE_DEF) return false;
            if (x.getType() == TokenTypes.ANNOTATION_DEF) return false;
            if (x.getType() == TokenTypes.ENUM_DEF) return false;
        }
        return false;
    }

    /**
     * Returns whether a node is contained in an outer class definition.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isInOuterMostClass(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.CLASS_DEF) return ScopeUtil.isOuterMostType(x);
            if (x.getType() == TokenTypes.INTERFACE_DEF) return false;
            if (x.getType() == TokenTypes.ANNOTATION_DEF) return false;
            if (x.getType() == TokenTypes.ENUM_DEF) return false;
        }
        return false;
    }

    /**
     * Returns whether a node is contained in a static method.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isInStaticMethod(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF) {
                return isStaticMethod(x);
            }
        }
        return false;
    }

    /**
     * Returns whether a node is contained in the main() method.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isInMainMethod(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF) {
                return isMainMethod(x);
            }
        }
        return false;
    }

    /**
     * Returns whether a node is the main() method.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isMainMethod(DetailAST ast) {
        if (ast.getType() != TokenTypes.METHOD_DEF) {
            return false;
        }

        // reject if not a method in outermost class
        boolean isOuter = ScopeUtil.isOuterMostType(ast.getParent().getParent());
        boolean isClass = (ast.getParent().getParent().getType() == TokenTypes.CLASS_DEF);
        if (!isOuter || !isClass) return false;

        // check name of method
        DetailAST ident = ast.findFirstToken(TokenTypes.IDENT);
        if (!"main".equals(ident.getText())) return false;

        // public and static
        DetailAST modifiersAST = ast.findFirstToken(TokenTypes.MODIFIERS);
        // if (!modifiersAST.branchContains(TokenTypes.LITERAL_PUBLIC)) return false;
        // if (!modifiersAST.branchContains(TokenTypes.LITERAL_STATIC)) return false;
        boolean isPublic = modifiersAST.findFirstToken(TokenTypes.LITERAL_PUBLIC) != null;
        boolean isStatic = modifiersAST.findFirstToken(TokenTypes.LITERAL_STATIC) != null;
        if (!isPublic) return false;
        if (!isStatic) return false;


        // return type is void
        DetailAST type = ast.findFirstToken(TokenTypes.TYPE).getFirstChild();
        if (type.getType() != TokenTypes.LITERAL_VOID) return false;

        // one argument of type String[] or String...
        final DetailAST params = ast.findFirstToken(TokenTypes.PARAMETERS);
        if (params.getChildCount() != 1) return false;
        DetailAST parameterType = params.getFirstChild().findFirstToken(TokenTypes.TYPE);
        Optional<DetailAST> arrayDecl = Optional.fromNullable(
                parameterType.findFirstToken(TokenTypes.ARRAY_DECLARATOR));
        Optional<DetailAST> varargs = Optional.fromNullable(
                params.getFirstChild().findFirstToken(TokenTypes.ELLIPSIS));

        if (arrayDecl.isPresent()) {
           return isStringType(arrayDecl.get().getFirstChild());
        }
        else if (varargs.isPresent()) {
            return isStringType(parameterType.getFirstChild());
        }
        else {
            return false;
        }
    }

    /**
     * Returns whether a node is contained in a hashCode() method.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isInHashCodeMethod(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF) {
                return isHashCodeMethod(x);
            }
        }
        return false;
    }

   // see checkstyle/code/MagicNumberCheck.java
    /**
     * Determines whether or not the given AST is a hashCode() method.
     * A valid hash code method is considered to be a method of the signature
     * {@code public int hashCode()}.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isHashCodeMethod(DetailAST ast) {
        if (ast.getType() != TokenTypes.METHOD_DEF) {
            return false;
        }

        // check name of method
        DetailAST ident = ast.findFirstToken(TokenTypes.IDENT);
        if (!"hashCode".equals(ident.getText())) return false;

        // no arguments
        final DetailAST params = ast.findFirstToken(TokenTypes.PARAMETERS);
        return params.getChildCount() == 0;
    }

    /**
     * Returns whether a node is contained in a toString() method.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isInToStringMethod(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF) {
                return isToStringMethod(x);
            }
        }
        return false;
    }

    /**
     * Determines whether or not the given AST is a toString() method.
     * A valid toString() method is considered to be a method of the signature
     * {@code public String toString()}.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public static boolean isToStringMethod(DetailAST ast) {
        if (ast.getType() != TokenTypes.METHOD_DEF) {
            return false;
        }

        // check name of method
        DetailAST ident = ast.findFirstToken(TokenTypes.IDENT);
        if (!"toString".equals(ident.getText())) return false;

        // no arguments
        final DetailAST params = ast.findFirstToken(TokenTypes.PARAMETERS);
        return params.getChildCount() == 0;
    }

}
