/*****************************************************************************
 *  Issues
 *  - checks only the name of the method/constructor
 *    (false positives if method has name name but differerent arguments or in different class)
 *  - checks only direct calls to methods in main(), not indirect calls
 *  - checks only public methods in outermost class
 *     (e.g., doesn't check next() and hasNext() in Iterator)
 *  - toString() is hard to deal with
 *  - considers a foreach loop to be an implicit call to iterator() method
 *****************************************************************************/

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Set;
import java.util.TreeSet;


public class MainCallsAllPublicMethodsCheck extends AbstractCheck {

    // the public methods
    private TreeSet<String> publicMethods;

    // the public methods
    private TreeSet<String> publicConstructors;

    // the public methods called in main() (or other static methods)
    private TreeSet<String> publicMethodsCalled;

    // the public constructors called in main() (or other static methods)
    private TreeSet<String> publicConstructorsCalled;

    // set of method names to ignore
    private final Set<String> ignoredMethodNames = new TreeSet<>();

    // The AST for the main() method; null if no such method.
    private DetailAST mainAST = null;

    // should we consider a call from any static methods to be a unit test?
    private boolean allowUnitTestsFromAnyStaticMethod = false;

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_METHOD = "main.calls.all.public.methods";
    public static final String MSG_CTOR = "main.calls.all.public.constructors";

    /**
     * Set the list of ignore method names.
     * @param methodNames array of ignored method names
     */
    public void setIgnoredMethodNames(String... methodNames) {
        ignoredMethodNames.clear();
        for (String methodName : methodNames) {
            ignoredMethodNames.add(methodName + "()");
        }
    }

    /**
     * Should we consider a call from any static method to be a unit test?
     * @param value true if we consider calls from other static methods to be unit tests
     */
    public void setAllowUnitTestsFromAnyStaticMethod(boolean value) {
        allowUnitTestsFromAnyStaticMethod = value;
    }

    @Override
    public int[] getDefaultTokens() {
        return new int[] {
            TokenTypes.METHOD_DEF,
            TokenTypes.CTOR_DEF,
            TokenTypes.METHOD_CALL,
            TokenTypes.LITERAL_NEW,
            TokenTypes.FOR_EACH_CLAUSE,
        };
    }

    @Override
    public int[] getRequiredTokens() {
        return getDefaultTokens();
    }

    @Override
    public int[] getAcceptableTokens() {
        return getDefaultTokens();
    }


    @Override
    public void visitToken(DetailAST ast) {

        // not in an outermost class
        if (!Utilities.isInOuterMostClass(ast)) {
            return;
        }

        // identify all public methods in outermost class (except main())
        if (ast.getType() == TokenTypes.METHOD_DEF) {
            if (Utilities.isMainMethod(ast)) {
                mainAST = ast;
                return;
            }
            if (!Utilities.isPublicMethod(ast)) return;
            DetailAST identifier = ast.findFirstToken(TokenTypes.IDENT);
            String methodName = identifier.getText() + "()";
            publicMethods.add(methodName);
        }

        // identify all public constructors in outermost class
        else if (ast.getType() == TokenTypes.CTOR_DEF) {
            if (!Utilities.isPublicConstructor(ast))
                return;
            DetailAST identifier = ast.findFirstToken(TokenTypes.IDENT);
            String constructorName = identifier.getText() + "()";
            publicConstructors.add(constructorName);
        }

        // identify all methods called directly in main()
        else if (ast.getType() == TokenTypes.METHOD_CALL) {
            if (!Utilities.isInStaticMethod(ast))
                return;
            if (!Utilities.isInMainMethod(ast) && !allowUnitTestsFromAnyStaticMethod)
                return;
            DetailAST dot = ast.findFirstToken(TokenTypes.DOT);
            if (dot == null) dot = ast;
            DetailAST identifier = Utilities.findLastToken(dot, TokenTypes.IDENT);
            String methodName = identifier.getText() + "()";
            publicMethodsCalled.add(methodName);

            // special case
            if (methodName.equals("sort()")) publicMethodsCalled.add("compareTo()");
        }

        // a foreach clause is an implicit call to iterator()
        else if (ast.getType() == TokenTypes.FOR_EACH_CLAUSE) {
            if (!Utilities.isInStaticMethod(ast))
                return;
            if (!Utilities.isInMainMethod(ast) && !allowUnitTestsFromAnyStaticMethod)
                return;
            String methodName = "iterator()";
            publicMethodsCalled.add(methodName);
        }

        // identify all constructors called directly in main()
        else if (ast.getType() == TokenTypes.LITERAL_NEW) {
            if (!Utilities.isInStaticMethod(ast))
                return;
            if (!Utilities.isInMainMethod(ast) && !allowUnitTestsFromAnyStaticMethod)
                return;
            DetailAST dot = ast.findFirstToken(TokenTypes.DOT);
            if (dot == null) dot = ast;
            DetailAST identifier = Utilities.findLastToken(dot, TokenTypes.IDENT);

            // probably an array
            if (identifier == null) return;

            String constructorName = identifier.getText() + "()";
            publicConstructorsCalled.add(constructorName);
        }

    }


    @Override
    public void beginTree(DetailAST rootAST) {
        publicMethods = new TreeSet<String>();
        publicMethodsCalled = new TreeSet<String>();
        publicConstructors = new TreeSet<String>();
        publicConstructorsCalled = new TreeSet<String>();
        mainAST = null;
    }

    @Override
    public void finishTree(DetailAST rootAST) {

        // check public methods
        ArrayList<String> publicMethodsNotCalled = new ArrayList<String>();
        for (String methodName : publicMethods) {
            if (!publicMethodsCalled.contains(methodName) && !ignoredMethodNames.contains(methodName)) {
                publicMethodsNotCalled.add(methodName);
            }
        }

        for (String methodName : publicMethodsNotCalled) {
            DetailAST ast = rootAST;

            // use identifier main() for line and column numbers
            if (mainAST != null) ast = mainAST.findFirstToken(TokenTypes.IDENT);

            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_METHOD,
                methodName);
        }

        // check public constructors
        ArrayList<String> publicConstructorsNotCalled = new ArrayList<String>();
        for (String constructorName : publicConstructors) {
            if (!publicConstructorsCalled.contains(constructorName) && !ignoredMethodNames.contains(constructorName))
                publicConstructorsNotCalled.add(constructorName);
        }

        for (String constructorName : publicConstructorsNotCalled) {
            DetailAST ast = rootAST;

            // use identifier main() for line and column numbers
            if (mainAST != null) ast = mainAST.findFirstToken(TokenTypes.IDENT);

            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_CTOR,
                constructorName);
        }
    }

}
