// number of instance variables of outer class
// (does not count inner class instance variables)

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;
import com.puppycrawl.tools.checkstyle.utils.ScopeUtil;
import java.util.ArrayList;

public class InstanceVariableCountCheck extends AbstractCheck {
    private int count = 0;                        // number of instance variables detected
    private int min = 0;                          // minimum number required
    private int max = Integer.MAX_VALUE;          // maximum number allowed

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_KEY = "instance.variable.count";

    // the instance variables
    private ArrayList<String> instanceVariables;

    @Override
    public int[] getDefaultTokens() {
        return new int[] {TokenTypes.VARIABLE_DEF};
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] {TokenTypes.VARIABLE_DEF};
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] {TokenTypes.VARIABLE_DEF};
    }

    /**
      * Sets a maximum count.
      * @param max the maximum count.
      */
    public void setMax(int max) {
        this.max = max;
    }

    /**
      * Sets a minimum count.
      * @param min the minimum count.
      */
    public void setMin(int min) {
        this.min = min;
    }

    @Override
    public void visitToken(DetailAST ast) {
        if (isInstanceVariableOfOutermostClass(ast)) {
            instanceVariables.add(findVariableName(ast));
            count++;
        }
    }

    @Override
    public void beginTree(DetailAST rootAST) {
        count = 0;
        instanceVariables = new ArrayList<String>();
    }

    @Override
    public void finishTree(DetailAST rootAST) {
        if (count > this.max || count < this.min) {
            // remove square brackets
            String variableList = instanceVariables.toString();
            variableList = variableList.substring(1, variableList.length() - 1);
            log(rootAST.getLineNo(),
                MSG_KEY,
                count,
                variableList);
        }
    }

    // extract and return the variable name from the AST
    private String findVariableName(DetailAST ast) {
        DetailAST identifier = ast.findFirstToken(TokenTypes.IDENT);
        // return identifier.toString();
        return "'" + identifier.getText() + "'";
    }

    // is the variable an instance variable of an outer class?
    private boolean isInstanceVariableOfOutermostClass(DetailAST ast) {
        if (ast.getType() != TokenTypes.VARIABLE_DEF) {
            throw new IllegalArgumentException("argument is not a VARIABLE_DEF");
        }
        boolean isInstanceVariable = Utilities.isInstanceVariable(ast);
        boolean isOuter = ScopeUtil.isOuterMostType(ast.getParent().getParent());
        return isInstanceVariable && isOuter;
    }

}
