// Check that number of tokens is within the specified min/max limits

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;
import com.puppycrawl.tools.checkstyle.utils.TokenUtil;
import com.puppycrawl.tools.checkstyle.utils.CommonUtil;
import java.util.TreeMap;

public class TokenCountCheck extends AbstractCheck {
    private int min = 0;                           // minimum number required
    private int max = Integer.MAX_VALUE;           // maximum number allowed
    private TreeMap<String, Integer> counts;       // number of occurrences of each token
    private int cumulativeCount = 0;               // total number of occurrences of all tokens
    private boolean sumTokenCounts = false;        // sum the token counts?
    private String callingMethodPattern = null;    // the calling method regexp to match (default = everything, even non methods)

    /**
     * Keys pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_MIN = "token.count.min";
    public static final String MSG_MAX = "token.count.max";
    public static final String MSG_MAX_CUMULATIVE = "token.count.cumulative.max";
    public static final String MSG_MIN_CUMULATIVE = "token.count.cumulative.min";

    @Override
    public int[] getDefaultTokens() {
        return CommonUtil.EMPTY_INT_ARRAY;
    }

    @Override
    public int[] getAcceptableTokens() {
        return TokenUtil.getAllTokenIds();
    }

    @Override
    public int[] getRequiredTokens() {
        return CommonUtil.EMPTY_INT_ARRAY;
    }

    @Override
    public boolean isCommentNodesRequired() {
        return true;
    }


    /**
      * Sets a maximum count.
      * @param max the maximum count.
      */
    public void setMax(int max) {
        this.max = max;
    }

    /**
      * Sets a minimum count.
      * @param min the minimum count.
      */
    public void setMin(int min) {
        this.min = min;
    }

    /**
     * Sets whether to use the sum of the token counts,
     * rather than the individual token counts.
     * @param sum whether to use the sum.
     */
    public void setSumTokenCounts(boolean sum) {
        sumTokenCounts = sum;
    }


    /**
     * Set the regexp for the calling method
     * @param callingMethodPattern the regexp for the calling method
     */
    public final void setCallingMethodPattern(String callingMethodPattern) {
        this.callingMethodPattern = callingMethodPattern;
    }


    /**
     * Returns whether a node is contained in the calling method.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public boolean isInCallingMethod(DetailAST ast) {
        if (callingMethodPattern == null) return true;
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF) {
                DetailAST ident = x.findFirstToken(TokenTypes.IDENT);
                return ident.getText().matches(callingMethodPattern);
            }
        }
        return false;
    }


    @Override
    public void visitToken(DetailAST ast) {
        // ignore unless we're in the calling method (or no calling method is specified)
        if (!isInCallingMethod(ast)) return;

        String token = TokenUtil.getTokenName(ast.getType());
        counts.put(token, counts.get(token) + 1);
            cumulativeCount++;

        // flag max violations at point of discovery
        int count = counts.get(token);
        if (!sumTokenCounts && (count > this.max)) {
            log(ast.getLineNo(),
                MSG_MAX,
                token,
                count,
                max);
        }

        // flag max cumulative violations at point of discovery
        if (sumTokenCounts && (cumulativeCount > this.max)) {
            log(ast.getLineNo(),
                MSG_MAX_CUMULATIVE,
                "{ " + getTokenNames().toString().replaceAll("[\\[\\]]", "") + " }",
                cumulativeCount,
                max);
        }
    }

    @Override
    public void beginTree(DetailAST rootAST) {
        counts = new TreeMap<String, Integer>();
        for (String token : getTokenNames()) {
            counts.put(token, 0);
        }
        cumulativeCount = 0;
    }

    // can't flag violations of min counts until entire tree is traversed
    @Override
    public void finishTree(DetailAST rootAST) {

        // check cumulative counts
        if (sumTokenCounts) {
            if (cumulativeCount < this.min) {
                log(rootAST.getLineNo(),
                    MSG_MIN_CUMULATIVE,
                    "{ " + getTokenNames().toString().replaceAll("[\\[\\]]", "") + " }",
                    cumulativeCount,
                    min);
            }
        }

        // check individual counts
        else {
            for (String token : getTokenNames()) {
                int count = counts.get(token);
                if (count < this.min) {
                    log(rootAST.getLineNo(),
                        MSG_MIN,
                        token,
                        count,
                        min);
                }
            }
        }
    }
}
